/*
 * aud_diaglib_dbus.cpp
 *
 *  Created on: Mar 15, 2016
 *      Author: rjk2kor
 */

#include "aud_diaglib_dbus.h"
#include "AudioStack/clGeniviAudioCtrlAdapter.h"
#include "AudioStack/AudioSources/clFactory_AudioSourceClass.h"
#include "AudioStack/AudioSources/clGeniviAudioSource.h"
#include "AudioStack/AudioSources/clAudioSourceFactory.h"
#include "AudioStack/SMT/clSrcStateFactory.h"
#include "CAmControlSenderBase.h"

#include "controllerplugin_Trace.h"
#ifndef USE_DLT_TRACE
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_AMCONTROLLERPLUGIN_DIAGNOSIS
#include "trcGenProj/Header/aud_diaglib_dbus.cpp.trc.h"
#endif

using namespace AudioStack;
using namespace AudioStack::AudioSource;
using namespace SourceStateMachine;

bool diaglib_dbus::bIsApplicable(std::string& rfSysSetId, std::string& rfSyssetType)
{
  if((std::string("DIA_SYSSET_ID_AUDIO_MANAGER") == rfSysSetId)||(std::string("DIA_SYSSET_ID_ALL_DOMAINS") == rfSysSetId))
  {
    if(
      (std::string("DIA_SYSSET_TYPE_PRODUCTION") == rfSyssetType)
      ||(std::string("DIA_SYSSET_TYPE_CUSTOMER") == rfSyssetType)
      ||(std::string("DIA_SYSSET_TYPE_HMI") == rfSyssetType)
      ||(std::string("DIA_SYSSET_TYPE_HMI_AUDIO") == rfSyssetType)
      )
    {
      return true;
    }
  }
  return false;
}

/**
   This is the constructor of the class diaglib_dbus

   @param[in] x Pointer to opaque DBusConnection object
   @return None
*/
diaglib_dbus::diaglib_dbus(DBusConnection* x,CAmControlSenderBase* y)
:aud_diag_defset_adaptor_IF::aud_diag_defset_adaptor_IF(x)
,m_poSenderBase(y)
{
  NORMAL_M_ASSERT(m_poSenderBase != NULL);
}
/**
   This is the destructor of the class diaglib_dbus
   @return None
*/
diaglib_dbus::~diaglib_dbus()
{
  m_poSenderBase = NULL;
}
/**
   This function is remotely invoked by diagnosis to trigger the component to prepare for system setting.

   @param[in] rfSysSetId Sys set ID
   @param[in] rfSyssetType Sys set Type
   @return Result of the operation
*/
uint32_t diaglib_dbus::u32OnPrepareSystemSetting(std::string& rfSysSetId, std::string& rfSyssetType)
{
  //As of now I am not considering the values of SysSetID and SysSetType to perform defset.
  //This checks can be added later

  ETG_TRACE_USR3(("Prepare Phase: Received SysSetID: %s",rfSysSetId.c_str()));
  ETG_TRACE_USR3(("Prepare Phase: Received SysSetType : %s",rfSyssetType.c_str()));

  if(!bIsApplicable(rfSysSetId,rfSyssetType))
  {
    return DIA_E_SUCCESS;
  }

    const clSourceClass* pSrcClass_MutefactoryResetOn = clFactory_AudioSourceClass::GetSourceClass("MUTE_FACTORYRESET");
    if (pSrcClass_MutefactoryResetOn == NULL)
    {
    ETG_TRACE_ERR(("u32OnPrepareSystemSetting(): MUTE_FACTORYRESET not configured in project, Defset Sequence not covered by MUTE ! ! !"));
    }
  else
  {
    clGeniviAudioCtrlAdapter::RequestSourceOn(SourceID(pSrcClass_MutefactoryResetOn->getClassID(), 0));
    ETG_TRACE_USR3(("Prepare Phase: Added Source to stack : MUTE_FACTORYRESET"));
  }
  return DIA_E_SUCCESS;
}
/**
   This function is remotely invoked by diagnosis to trigger the component to prepare for system setting.

   @param[in] rfSysSetId Sys set ID
   @param[in] rfSyssetType Sys set Type
   @return Result of the operation
*/
uint32_t diaglib_dbus::u32OnExecuteSystemSetting(std::string& rfSysSetId, std::string& rfSyssetType)
{
  //As of now I am not considering the values of SysSetID and SysSetType to perform defset.
  //This checks can be added later
  ETG_TRACE_USR3(("Execute Phase: Received SysSetID: %s",rfSysSetId.c_str()));
  ETG_TRACE_USR3(("Execute Phase: Received SysSetType : %s",rfSyssetType.c_str()));

  if(!bIsApplicable(rfSysSetId,rfSyssetType))
  {
    return DIA_E_SUCCESS;
  }

  ETG_TRACE_USR3(("u32OnExecuteSystemSetting(): Trying to activate default source : midw_fi_tcl_e8_AudSource::FI_EN_FM"));
  sourceClassID enInternalSrcClass = clGeniviAudioCtrlAdapter::Ext2Int_SrcClass(midw_fi_tcl_e8_AudSource::FI_EN_FM);
  const clSourceClass srcClass = clFactory_AudioSourceClass::GetSourceClass(enInternalSrcClass);
  clAudioSource* pAudioSource = AudioStack::AudioSource::clAudioSourceFactory::getAvailableAudioSource(srcClass.SourceClassID);

  if (pAudioSource)
  {
    SourceID SrcID = pAudioSource->sGetId();
    clGeniviAudioSource* pGamSource = clGeniviAudioCtrlAdapter::getAudioSource(SrcID);
    if (pGamSource != NULL)
    {
      tU16 u16GamSrc = pGamSource->u16GetGeniviSourceID();
      ETG_TRACE_USR3(("calling CAmControlSenderBase::  hookUserConnectionRequest with GAM Src %d", u16GamSrc));
      am_mainConnectionID_t mainConnectionID;
      if(m_poSenderBase)
      {
        m_poSenderBase->hookUserConnectionRequest(u16GamSrc, 1, mainConnectionID); //SinkID = 1
        ETG_TRACE_USR3(("u32OnExecuteSystemSetting(): FM Source put on Stack"));
      }
      else
      {
        ETG_TRACE_FATAL(("u32OnExecuteSystemSetting(): m_poSenderBase is NULL ! ! !"));
      }
    }
    else
    {
      ETG_TRACE_ERR(("u32OnFinalizeSystemSetting(): FM Not maaped to GeniviAudio Source !!! "));
    }
  }
  else
  {
    ETG_TRACE_ERR(("u32OnFinalizeSystemSetting(): FM source not configured in project ! ! !"));
  }
  return DIA_E_SUCCESS;
}
/**
   This function is remotely invoked by diagnosis to trigger the component to prepare for system setting.

   @param[in] rfSysSetId Sys set ID
   @param[in] rfSyssetType Sys set Type
   @return Result of the operation
*/
uint32_t diaglib_dbus::u32OnFinalizeSystemSetting(std::string& rfSysSetId, std::string& rfSyssetType)
{
  //As of now I am not considering the values of SysSetID and SysSetType to perform defset.
  //This checks can be added later

  ETG_TRACE_USR3(("Finalize Phase: Received SysSetID: %s",rfSysSetId.c_str()));
  ETG_TRACE_USR3(("Finalize Phase: Received SysSetType : %s",rfSyssetType.c_str()));

  if(!bIsApplicable(rfSysSetId,rfSyssetType))
  {
    return DIA_E_SUCCESS;
  }

    const clSourceClass* pSrcClass_MutefactoryResetOn = clFactory_AudioSourceClass::GetSourceClass("MUTE_FACTORYRESET");
    if (pSrcClass_MutefactoryResetOn == NULL)
    {
    ETG_TRACE_ERR(("u32OnFinalizeSystemSetting(): MUTE_FACTORYRESET not configured in project, Defset Sequence not covered by MUTE ! ! !"));
    }
  else
  {
    clGeniviAudioCtrlAdapter::RequestSourceOff(SourceID(pSrcClass_MutefactoryResetOn->getClassID(), 0));
    ETG_TRACE_USR3(("Finalize Phase: Removed Source from stack : MUTE_FACTORYRESET"));
  }
  return DIA_E_SUCCESS;
}


/**
   Constructor
*/
aud_diaglib_handler::aud_diaglib_handler()
:bStandalone(false)
,m_ptrConnection(NULL)
,m_pDiaglib(NULL)
,m_pWatchConnector(NULL)
,m_pTimerConnector(NULL)
{
}

/**
   This function is called from CAmControlSenderBase to initialize the diaglib handler.
   When this funtion is invoked the following steps are done
   1. If a stand alone dbus server is configured, then
  a)A private connection to dbus system bus is taken
  b)Timer and watches to the connection are setup for asynchronous handling and mainloop integration
  c)A pre configured bus name is requested on the new connection.
   2. If a shared connection is configured, then shared connection to the bus is requested.

   A object of diaglib_dbus class is created to take care of messages from diagnosis
   @return TRUE if the setup is successful, else false
*/

bool aud_diaglib_handler::Init(am::CAmSocketHandler* poMainLoop,CAmControlSenderBase* pSenderBase)
{
  if(poMainLoop == NULL)
    return false;

  if(m_pWatchConnector || m_ptrConnection || m_pTimerConnector)
  {
    ETG_TRACE_USR3(("aud_diaglib_handler::Init. Already initialized "));
    return true;
  }

#if !defined(GAM_DEFSET_SETUP_DBUS_SHARED) && !defined (GAM_DEFSET_SETUP_DBUS_STANDALONE)
    ETG_TRACE_USR3(("aud_diaglib_handler::Init. GAM configured to use diag system settings from CCA "));
  return false;
#endif
  ///Enable Multi thread handling in libdbus
  dbus_threads_init_default();
  ///We cannot use CAMDbusWrapper interface provided by genivi audio manager to get the dbus connection used by it,
  ///as it is not exposed to the controller plugin
  ///As of now, Genivi Audio Manager is using System bus. We will use system bus and try to get a shared connection.
  ///This is done so that, this object path would be registered on the same connection of GAM, for aesthetic reasons.
#ifdef GAM_DEFSET_SETUP_DBUS_STANDALONE
  bStandalone = true;
#endif

  if(bStandalone)
  {
    ETG_TRACE_USR3(("aud_diaglib_handler::Init. Trying to setup standalone dbus server "));
    m_ptrConnection = dbus_bus_get_private(DBUS_BUS_SYSTEM,NULL);
    if(m_ptrConnection != NULL)
    {
      //1. Attach watch to connection
      m_pWatchConnector = new aud_ml_dbus_watch_connector(m_ptrConnection,poMainLoop);
      //2. Attach timer to connection
      m_pTimerConnector = new aud_ml_dbus_timer_connector(m_ptrConnection,poMainLoop);
      //3. Request our own name for this connection
      dbus_bus_request_name(m_ptrConnection,AUD_GAM_DEFSET_STANDALONE_BUS_NAME,DBUS_NAME_FLAG_DO_NOT_QUEUE,NULL);
      //As this is a standalone connection, inform dbus to not trigger _exit() function on disconnect
      dbus_connection_set_exit_on_disconnect(m_ptrConnection,FALSE);
    }
  }
  else
  {
    ETG_TRACE_USR3(("aud_diaglib_handler::Init: Trying to setup interface on bus name org.genivi.audiomanager "));
    m_ptrConnection = dbus_bus_get(DBUS_BUS_SYSTEM,NULL);
  }

  if(m_ptrConnection == NULL)
  {
    ETG_TRACE_FATAL(("aud_diaglib_handler::Init. Couldn't get a connection to the BUS ! ! ! "));
    return false;
  }
  //Create object of diaglib dbus
  m_pDiaglib = new diaglib_dbus(m_ptrConnection,pSenderBase);

  return true;
}

/**
   This function is called from CAmControlSenderBase to deinitialize the diaglib handler.
   This function would close/unref the dbus connections and free all the resources previously allocated
*/

void aud_diaglib_handler::DeInit()
{
  if(m_ptrConnection)
  {
    if(bStandalone)
    {
      //Close required for private connections, not required for shared connections
      dbus_connection_close(m_ptrConnection);
    }
    //Release the reference to DBusConnection
    dbus_connection_unref(m_ptrConnection);
    m_ptrConnection = NULL;
  }
  if(m_pDiaglib)
  {
    delete m_pDiaglib;
    m_pDiaglib = NULL;
  }
  if(m_pWatchConnector)
  {
    delete m_pWatchConnector;
    m_pWatchConnector = NULL;
  }
  if(m_pTimerConnector)
  {
    delete m_pTimerConnector;
    m_pTimerConnector = NULL;
  }
}
/**
   Destructor
*/
aud_diaglib_handler::~aud_diaglib_handler()
{
  DeInit();
}
